/**
 * \file        DMNetMapper.hpp
 * \brief       class definition for (re)mapping DM Net control flow
 * \author      Larry Salant
 * \date        1/20/2009
 *
 */
#ifndef __DMNET_MAPPER_HPP__
#define __DMNET_MAPPER_HPP__

////////////////////////////////////////////////////////////////////////////////

#include "dm_nutypedefs.h"
#include "CresnetStruct.h"

////////////////////////////////////////////////////////////////////////////////

typedef enum
{
  DMNET_SLAVE = 0,  // indicates port is a slave in normal flow (also used as array index)
  DMNET_MIDPOINT,   // indicates port is a Midpoint slave in normal flow (also used as array index)
  DMNET_MASTER      // indicates port is a master (in normal control flow)
}DMNET_PORT;

typedef void (*DMMASTERFWDMSG)(CREST_ALL_PACKETS *packet);

class CDMCresnetUartHost;

class CDMNetMapper
{
  public:
    CDMNetMapper(void);

    virtual UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream) {return 0;}
    virtual UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream){return 0;}
		virtual void SendBroadcastToEndpoint(UINT8 *pPacket){}
    virtual void SendToEndpoint( UINT8* pPacket, UINT8 stream );
    virtual void ControllerRxPacketProcessing( UINT8 stream );

    virtual UINT8 SendPendingToController(UINT8 bStream){return 0;}
    virtual UINT8 IsControllerAvailable(UINT8 bStream){return 0;}
		virtual void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size){}
		virtual void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command){}

		virtual bool SetControlFlowReverse(void){return false;}
    virtual bool SetControlFlowNormal(void){return false;}
    virtual bool CanBeMidpoint(UINT8 Port){return false;}
    virtual void ReportOnlineStatus(void);
    virtual void MasterStatusChange(UINT8 bStream, bool bState);
    bool IsStandardFlow(void);

		void SetFwdMsgFunctionPointer( DMMASTERFWDMSG pfDmMasterFwdMsg )
    {
        m_pfDmMasterFwdMsg = pfDmMasterFwdMsg;
    }
	
	int m_iDmNet;
    UINT32 m_pDmNetUart;

  protected:
    UINT8 m_ControlFlow;
		
    DMMASTERFWDMSG m_pfDmMasterFwdMsg;
};

class CDMNetMapperSlaveOnly : public CDMNetMapper
{
  public:
    CDMNetMapperSlaveOnly(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);	
};


class CDMNetMapperStro : public CDMNetMapper
{
  public:
    CDMNetMapperStro(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
};


class CDMNetMapperStr : public CDMNetMapper
{
  public:
    CDMNetMapperStr(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
};



class CDMNetMapperRepeater : public CDMNetMapper
{
  public:
    CDMNetMapperRepeater(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void ReportOnlineStatus(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
};

class CDMNetMapperOutMidpoint : public CDMNetMapper
{
  public:
    CDMNetMapperOutMidpoint(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);
    void SendToEndpoint(UINT8 *pPacket, UINT8 stream);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void ReportOnlineStatus(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
};

class CDMNetMapperInMidpoint : public CDMNetMapper
{
  public:
    CDMNetMapperInMidpoint(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);    
};

class CDMNetMapperMidpointDeviceOutput : public CDMNetMapper
{
  public:
    CDMNetMapperMidpointDeviceOutput(int iDmNet = 1);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);
    void SendToEndpoint(UINT8 *pPacket, UINT8 stream);      

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
    void ControllerRxPacketProcessing( UINT8 stream );
};

class CDMNetMapperMidpointDeviceInput : public CDMNetMapper
{
  public:
    CDMNetMapperMidpointDeviceInput(int iDmNet = 1);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);    
    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
    void SendToEndpoint(UINT8 *pPacket, UINT8 stream);
    void ControllerRxPacketProcessing( UINT8 stream );
    
};
class CDMNetMapperSrcEndpoint : public CDMNetMapper
{
  public:
    CDMNetMapperSrcEndpoint(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);
    void SendToEndpoint( UINT8* pPacket, UINT8 stream );
    void ControllerRxPacketProcessing( UINT8 stream );

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
		
  private:
        UINT32 m_OfflineTimer;
};
class CDMNetMapperFiberSrcEndpoint : public CDMNetMapper
{
  public:
    CDMNetMapperFiberSrcEndpoint(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);
    void SendToEndpoint( UINT8* pPacket, UINT8 stream );
    void ControllerRxPacketProcessing( UINT8 stream );

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
};

class CDMNetMapperFiberOutMidpoint : public CDMNetMapper
{
  public:
    CDMNetMapperFiberOutMidpoint();

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void ReportOnlineStatus(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
};

class CDMNetMapperFiberInMidpoint : public CDMNetMapper
{
  public:
    CDMNetMapperFiberInMidpoint(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
};

class CDMNetMapperFiberSlave : public CDMNetMapper
{
  public:
    CDMNetMapperFiberSlave(void);

    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
};

class CDMNetMapperIRControllerOnly : public CDMNetMapper
{
private:
    UINT8 m_I2CBusNum;
  public:
    CDMNetMapperIRControllerOnly(UINT8);

    UINT8 IrProcessRxPacket(UINT8 *pData, UINT8 byteCount);
    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
};

class DMNetMapperS3IOP : public CDMNetMapper
{
private:
  public:
    DMNetMapperS3IOP(UINT8);

    UINT8 ProcessRxPacketPtr(UINT8 Port, UINT8 *pData, UINT8 byteCount);
    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
};

class DMNetMapperRF : public CDMNetMapper
{
private:
  public:
    DMNetMapperRF(); // Constructor

    UINT8 ProcessRxPacketPtr(UINT8 Port, UINT8 *pData, UINT8 byteCount);
    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    
    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool IsStandardFlow(void);
};

class CDMNetMapperIopOutMidpoint : public CDMNetMapper
{
  public:
    CDMNetMapperIopOutMidpoint(void);

    void ProcessRxPacketPtr(UINT8 *pPkt);
    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);
    void SendToEndpoint(UINT8 *pPacket, UINT8 stream);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
    void ReportOnlineStatus(void);
    void MasterStatusChange(UINT8 bStream, bool bState);
};

class CDMNetMapperIopSlaveOnly : public CDMNetMapper
{
  public:
    CDMNetMapperIopSlaveOnly(void);

    void ProcessRxPacketPtr(UINT8 *pPkt);
    UINT8 ProcessRxPacket(UINT8 Port, UINT8 Stream);
    UINT8 SendEncapToEndpoint(UINT8 *pPacket, UINT8 Stream);
    void SendBroadcastToEndpoint(UINT8 *pPacket);

    UINT8 SendPendingToController(UINT8 bStream);
    UINT8 IsControllerAvailable(UINT8 bStream);
    void QueuePacketToController(UINT8 bStream, void *packet, UINT32 packet_size);
    void QueFromMasterToController(CDMCresnetUartHost *pDmHost, UINT8 command);

    bool SetControlFlowReverse(void);
    bool SetControlFlowNormal(void);
    bool CanBeMidpoint(UINT8 Port);
    bool IsStandardFlow(void);
};

#define STANDARD_CONTROL_FLOW 0
#define REVERSE_CONTROL_FLOW 1

extern CDMNetMapper *pDMNetMapper;

CDMNetMapper *GetDMNetMapperFromCresnetId(UINT8 cresnetId);
CDMNetMapper *GetDMNetMapperFromStreamId(UINT8 streamId);
void InitializeDMNetMapperArray();


// timer to determine if Endpoint (or Repeater) connected to Midpoint Slave
// has gone off line
#define MIDPOINT_TIMER_PERIOD      100 // check timeout period (in ms)
#define ENDPOINT_OFFLINE_TIMEOUT   1000
#define ENDPOINT_WATCHDOG_TIMEOUT  (ENDPOINT_OFFLINE_TIMEOUT/MIDPOINT_TIMER_PERIOD) // timeout in ms converted to ticks

#endif










